# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Development Commands

### Building the Application
- **Dev build**: ./bin/build/app_dev.sh - Creates development build with mod tidy, vendor, and builds to app binary
- **Prod build**: ./bin/build/app_prod.sh - Production build
- **Live reload**: ./bin/run/app_web_dev.sh - Runs with Air for hot reloading using air.dev.toml config

### Running Tests
- **Single test**: go test ./tests/functional/path/to/specific_test.go - Run a specific test file
- **All tests**: ./bin/tests/check.sh - Cleans test cache and runs all tests in ./tests/...
- **IMPORTANT**: Always run the most relevant single test first (quick), then run all tests (slow)
- **If you change a test, always run it to ensure it works**
- **If you change a service code, always run the service test to ensure it works**

### CSS Development
- **Compile CSS**: npx tailwindcss -i input.css -o ./assets/css/output.css --watch - Compiles Tailwind CSS with watch mode

### Docker Development
- **Dev mode**: Uses Dockerfile with dev stage, runs build/init/run scripts
- **Prod mode**: Uses Dockerfile with prod stage for production deployment

## Architecture Overview

This is a Go-based web application built with:

### Core Framework
- **Gin** web framework for HTTP routing and middleware
- **SQLite3** database with optional PostgreSQL support
- **Tailwind CSS** for styling with live compilation
- **Air** for development hot reloading

### Application Structure

#### Main Entry Point
- main.go - Application entry point with panic recovery, builds and runs app based on ENV

#### Core Layers (src/):
- **app/** - Application bootstrapping, DI container, and router setup
- **mdl/** - Domain models
- **mae/** - Business logic modules following Maestro pattern
- **dat/** - Data structures (forms, constants, Maestro input/output types)
- **lib/** - Shared libraries and utilities
- **srv/** - Service layer (handlers, presenters, validators, fetchers, etc.)

#### Maestro Pattern
The application follows a Maestro pattern where:
- Each business operation has a dedicated Maestro module (e.g., CreateThingMae)
- MAEs contain steps that execute in sequence
- Steps handle specific concerns (validation, persistence, event recording)

#### Key Directories

**Core Application Structure:**
- **src/app/routing/** - Route definitions organized by type (api_routes/, app_routes/, ajx_routes/)
- **src/app/container/** - Dependency injection container and service registration
- **src/app/routers/** - Router setup and middleware configuration

**Business Logic & Models:**
- **src/mae/** - Maestro modules containing business logic steps
- **src/mdl/** - Domain models
- **src/dat/forms/** - Form data structures for user input
- **src/dat/maes/** - Input/output data structures for Maestro operations

**Service Layer (src/srv/):**
- **binders/** - Request data binding to form structures
- **handlers/** - Entity-specific CRUD operations and business logic
- **fetchers/** - Data retrieval services for entities
- **html_views/** - HTML view renderers for web pages
- **json_views/** - JSON API response formatters
- **form_validators/** - Input validation for forms
- **form_presenters/** - Form display logic and field configuration
- **presenters/** - Data presentation logic for entities
- **show_presenters/** - Detail view presentation logic
- **tables/** - Table/list view configuration
- **sections/** - Modular UI components (navbar, forms, lists per entity)
- **layouts/** - Page layout templates (structured, tiled, vertical, empty)
- **molders/** - Form data transformation and preparation
- **hydrators/** - Entity state management and data population
- **searchers/** - Search functionality implementation
- **completers/** - Auto-completion services
- **companions/** - Entity relationship management
- **serializers/** - Data serialization for API responses
- **stringers/** - String representation utilities
- **relationers/** - Entity relationship mapping
- **entity_validators/** - Domain-level validation rules
- **entity_policies/** - Access control and business rules
- **illustrers/** - Display formatting (search results, table columns)
- **commands/** - Application commands (web server, data loading)
- **urlers/** - URL generation utilities
- **menus/** - Navigation menu configuration

**Shared Libraries (src/lib/):**
- **maestro/** - Core Maestro pattern implementation
- **sql/** - Database abstraction layer (PostgreSQL, SQLite)
- **middlewares/** - HTTP middlewares (JWT, session, recovery, dev tools)
- **form/** - Form handling utilities
- **presenter/** - Base presentation components
- **router/** - Routing utilities and middleware
- **error/** - Error handling utilities
- **logger/** - Logging infrastructure
- **time/** - Time utilities with deterministic implementations for testing
- **operating_system/** - OS abstraction layer
- **validator/** - Validation utilities
- **config/** - Configuration management

**Frontend Assets:**
- **assets/css/** - Compiled Tailwind CSS and custom styles
- **assets/js/** - JavaScript utilities and Tailwind configuration
- **assets/components/** - Reusable frontend components (forms, search, tables)
- **assets/libs/** - Third-party libraries (Alpine.js, Tailwind)

**Development & Operations:**
- **bin/** - Build, run, test, and deployment scripts
- **config/envs/** - Environment configuration files
- **data/** - Database files and schema definitions
- **tests/functional/** - Integration tests for MAEs, views, and routers
- **tools/** - Development utilities (CLI tools, sync scripts)
- **sup/** - Support libraries for testing and development

#### Database
- SQLite3 primary database with schema in data/schema.sqlite.sql
- PostgreSQL schema available in data/schema.pg.sql
- Database operations abstracted through sql_db library

#### Testing
- Functional tests in tests/functional/ covering Maestros, routers, and views
- Uses Go's standard testing framework

## MCP Integration

This project uses Model Context Protocol (MCP) to integrate with Claude Code. The ExchangeRouter has been enhanced with MCP-compliant endpoints for automated model generation and code patching.

### MCP Workflow for Model Creation

**IMPORTANT**: When you detect the need to create a new model, follow this workflow instead of directly coding:

1. **Detect Model Creation**: Look for requests like "create a new model", "add model", "generate model for X"
2. **Use MCP Tools**: Instead of writing code directly, use the available MCP tools:
   - Use "create_model" tool to create a new model via the ExchangeRouter
   - Use "apply_patch" tool to generate and retrieve code patches via the ExchangeRouter
3. **Apply Changes**: The ExchangeRouter will handle model creation and patch generation through the existing endpoints

### MCP Configuration

The MCP integration is configured in ".mcp.json" via "/home/cyrille/mcp/gen_x_mcp_stdio" and connects directly to the ExchangeRouter at "http://localhost:7086/exchange/mcp".

	- "POST /exchange/mcp" - Main MCP protocol handler
	- "GET /exchange/mcp/tools" - List available MCP tools

	Ensure the gen-x application is running before using MCP tools.

	### Available MCP Tools

	- **create_model**: Creates a new model using the ExchangeRouter's "/exchange/create-model" endpoint
	- **apply_patch**: Generates patches using the ExchangeRouter's "/exchange/patch" endpoint

## Development Workflow

1. **Environment Setup**: ENV=dev for development, ENV=prod for production
2. **Live Development**: Use ./bin/run/app_web_dev.sh for hot reloading
3. **CSS Changes**: Run Tailwind watch command for automatic CSS compilation
4. **Testing**: Run ./bin/tests/check.sh before commits
5. **Database**: SQLite database file at data/db.db