# Framework Documentation

A comprehensive Go web application framework featuring the Maestro pattern, complete service architecture, and rapid development tooling.

## Navigation Menu

### Getting Started
**Start here if you're new to the framework**

| Topic | Description | Key Files |
|-------|-------------|-----------|
| **[Installation](getting-started/installation.md)** | Setup, dependencies, environment | Requirements, Docker, local setup |
| **[Configuration](getting-started/configuration.md)** | Environment variables, database setup | Development, production, testing configs |
| **[Quickstart](getting-started/quickstart.md)** | Your first entity in 10 minutes | Step-by-step walkthrough |
| **[First Application](getting-started/first-application.md)** | Complete application tutorial | User management system |

### The Basics
**Essential concepts for building applications**

| Topic | Description | When to Use |
|-------|-------------|-------------|
| **[Common Patterns](the-basics/common-patterns.md)** | Core framework patterns | Understanding data flow |
| **[Routing](the-basics/routing.md)** | URL routing, middleware | Setting up endpoints |
| **[Controllers](the-basics/controllers.md)** | Request handling | Processing HTTP requests |
| **[Models](the-basics/models.md)** | Domain entities | Defining business objects |
| **[Forms](the-basics/forms.md)** | Input handling, validation | User data processing |
| **[Views](the-basics/views.md)** | HTML/JSON responses | Presenting data |
| **[Database](the-basics/database.md)** | Queries, transactions, migrations | Data persistence |
| **[Configuration](the-basics/configuration.md)** | Settings management | App configuration |
| **[Frontend](the-basics/frontend.md)** | CSS, JavaScript, assets | UI development |
| **[Error Handling](the-basics/error-handling.md)** | Error patterns, recovery | Robust error management |

### Architecture
**Framework design and core patterns**

| Topic | Description | Key Concepts |
|-------|-------------|--------------|
| **[Overview](architecture/overview.md)** | Framework architecture | Service layers, data flow |
| **[Maestro Pattern](architecture/maestro-pattern.md)** | Business logic orchestration | Mae modules, steps |
| **[Service Layer](architecture/service-layer.md)** | Service types and roles | Fetchers, Handlers, Hydrators |
| **[Dependency Injection](architecture/dependency-injection.md)** | Container, service wiring | DI patterns |
| **[Request Lifecycle](architecture/request-lifecycle.md)** | Request processing flow | Middleware, routing |

### Services
**Complete service layer documentation**

| Service Type | Purpose | Documentation |
|--------------|---------|---------------|
| **[Fetchers](services/fetcher.md)** | Data retrieval, querying | Filtering, pagination, scoping |
| **[Handlers](services/handlers.md)** | CRUD operations | Create, read, update, delete |
| **[Hydrators](services/hydrators.md)** | Relationship loading | Batch loading, presets |
| **[Molders](services/molders.md)** | Data transformation | Form-to-entity conversion |
| **[Relationers](services/relationers.md)** | Association management | Entity relationships |
| **[Searchers](services/searchers.md)** | Search functionality | Full-text, indexing |
| **[Serializers](services/serializers.md)** | JSON formatting | API responses |
| **[Stringers](services/stringers.md)** | String conversion | Human-readable output |
| **[Tables](services/tables.md)** | Database mapping | ESO pattern, SQL mapping |
| **[Binders](services/binders.md)** | Request binding | HTTP data extraction |
| **[Presenters](services/presenters.md)** | Display formatting | Column management, views |
| **[Form Validators](services/form_validators.md)** | Input validation | Form data validation |
| **[Entity Validators](services/entity_validators.md)** | Business rule validation | Domain validation |
| **[HTML Views](services/html_views.md)** | Page generation | Layout assembly |
| **[JSON Views](services/json_views.md)** | API responses | Response serialization |

### Advanced
**Complex patterns and optimization**

| Topic | Description | Use Cases |
|-------|-------------|-----------|
| **[Middleware](advanced/middleware.md)** | Request processing | Auth, CORS, rate limiting |
| **[Caching](advanced/caching.md)** | Performance optimization | Response caching, data caching |
| **[Security](advanced/security.md)** | Application security | Authentication, authorization |
| **[Performance](advanced/performance.md)** | Optimization strategies | Database, memory, response times |

### Development
**Testing, debugging, and deployment**

| Topic | Description | Tools |
|-------|-------------|-------|
| **[Testing](development/testing.md)** | Testing patterns | Unit, functional, integration |
| **[Debugging](development/debugging.md)** | Debugging techniques | Logging, profiling |
| **[Deployment](development/deployment.md)** | Production deployment | Docker, CI/CD |

### Deeper
**Internals and advanced customization**

| Topic | Description | For Advanced Users |
|-------|-------------|-------------------|
| **[Internals](deeper/internals.md)** | Framework internals | Understanding core mechanisms |
| **[Extending](deeper/extending.md)** | Framework extension | Custom services, patterns |
| **[Patterns](deeper/patterns.md)** | Advanced patterns | Custom implementations |

### Libraries
**Complete API reference for framework libraries**

*Library documentation provides cross-references to main documentation sections*

| Category | Libraries | Main Documentation |
|----------|-----------|-------------------|
| **Core** | config, error, logger, validator | [Configuration](getting-started/configuration.md), [Error Handling](the-basics/error-handling.md) |
| **Data** | fetcher, handler, hydrator, sql | [Database](the-basics/database.md), [Services](services/) |
| **Web** | router, middlewares, form | [Routing](the-basics/routing.md), [Forms](the-basics/forms.md) |
| **Presentation** | presenter, binder | [Views](the-basics/views.md), [Services](services/) |
| **Business** | maestro, validator | [Maestro Pattern](architecture/maestro-pattern.md) |

---

## Quick Start Paths

### I'm New to the Framework
1. **[Installation](getting-started/installation.md)** - Get your environment ready
2. **[Quickstart](getting-started/quickstart.md)** - Build your first entity
3. **[Common Patterns](the-basics/common-patterns.md)** - Understand the core patterns
4. **[Database](the-basics/database.md)** - Learn data persistence

### I Want to Build Features
1. **[Models](the-basics/models.md)** - Define your business objects
2. **[Maestro Pattern](architecture/maestro-pattern.md)** - Structure your business logic
3. **[Services Overview](services/)** - Choose the right services
4. **[Forms](the-basics/forms.md)** & **[Views](the-basics/views.md)** - Handle input and output

### I Need to Integrate Systems
1. **[Middleware](advanced/middleware.md)** - Handle cross-cutting concerns
2. **[Security](advanced/security.md)** - Authentication and authorization
3. **[JSON Views](services/json_views.md)** - Build APIs
4. **[Caching](advanced/caching.md)** - Optimize performance

### I Want to Deploy to Production
1. **[Configuration](getting-started/configuration.md)** - Environment setup
2. **[Performance](advanced/performance.md)** - Optimization strategies
3. **[Testing](development/testing.md)** - Ensure quality
4. **[Deployment](development/deployment.md)** - Go live

---

## Framework Architecture

### Core Technologies Stack
| Technology | Purpose | Documentation |
|------------|---------|---------------|
| **Go** | Primary language, strong typing | [Getting Started](getting-started/) |
| **Gin** | HTTP routing, middleware | [Routing](the-basics/routing.md), [Middleware](advanced/middleware.md) |
| **SQLite3/PostgreSQL** | Database layer | [Database](the-basics/database.md) |
| **Tailwind CSS** | Utility-first styling | [Frontend](the-basics/frontend.md) |
| **Air** | Hot reloading development | [Installation](getting-started/installation.md) |

### Key Patterns
| Pattern | Description | Learn More |
|---------|-------------|------------|
| **Maestro Pattern** | Business logic orchestration with sequential steps | [Architecture](architecture/maestro-pattern.md) |
| **Service Layer** | Comprehensive service types for different concerns | [Service Layer](architecture/service-layer.md) |
| **Dependency Injection** | Container-based service management | [DI Container](architecture/dependency-injection.md) |
| **ESO Pattern** | Entity Storage Objects for database mapping | [Tables](services/tables.md) |

### Directory Structure
```
src/
├── app/           # Application bootstrapping, DI container → [Architecture](architecture/)
├── mdl/           # Domain models → [Models](the-basics/models.md)
├── mae/           # Maestro business logic → [Maestro Pattern](architecture/maestro-pattern.md)
├── dat/           # Data structures, forms → [Forms](the-basics/forms.md)
├── lib/           # Shared libraries → [Libraries](libraries/)
└── srv/           # Service layer → [Services](services/)
```

---

## Common Development Workflows

### Creating a New Feature
1. **Define** your domain model in `src/mdl/`
2. **Map** database structure in `src/srv/tables/`
3. **Generate** service suite in `src/srv/`
4. **Build** business logic in `src/mae/`
5. **Create** forms in `src/dat/forms/`
6. **Wire** routes in appropriate router

**Learn more:** [First Application Tutorial](getting-started/first-application.md)

### Adding Business Logic
1. **Create** Mae module in `src/mae/`
2. **Define** input/output in `src/dat/maes/`
3. **Implement** business steps
4. **Register** with DI container
5. **Wire** routes and controllers

**Learn more:** [Maestro Pattern](architecture/maestro-pattern.md)

### Working with Data
| Task | Service | Documentation |
|------|---------|---------------|
| **Query Data** | Fetchers | [Fetchers](services/fetcher.md) |
| **Modify Data** | Handlers | [Handlers](services/handlers.md) |
| **Load Relationships** | Hydrators | [Hydrators](services/hydrators.md) |
| **Transform Data** | Molders | [Molders](services/molders.md) |
| **Search Data** | Searchers | [Searchers](services/searchers.md) |

---

## Documentation Principles

This documentation is organized by **expertise level** and **use case**:

- **Conceptual** - Understanding patterns and architecture
- **Practical** - Step-by-step implementation guides
- **Reference** - Complete API and service documentation
- **Deep Dive** - Advanced topics and internals

### Finding What You Need
- **New to framework?** Start with [Getting Started](getting-started/)
- **Need to implement something?** Check [The Basics](the-basics/)
- **Want to understand architecture?** Read [Architecture](architecture/)
- **Looking for service details?** Browse [Services](services/)
- **Need advanced patterns?** Explore [Advanced](advanced/)
- **Debugging or deployment?** See [Development](development/)

---

**Ready to build?** Start with the [Getting Started Guide](getting-started/) and create your first application in minutes.