# Installation

## Prerequisites

### Required Software
- **Go 1.21+** - Primary programming language
- **Node.js 16+** - For Tailwind CSS compilation
- **Git** - Version control system

### Optional Tools
- **Docker** - For containerized development
- **PostgreSQL** - Alternative to SQLite for production
- **Air** - Hot reloading (installed automatically)

## Quick Installation

### 1. Clone and Setup
```bash
# Clone the repository
git clone <repository-url>
cd <project-name>

# Install Go dependencies
go mod tidy
go mod vendor

# Install Node.js dependencies for CSS compilation
npm install
```

### 2. Environment Configuration
```bash
# Set development environment
export ENV=dev

# Copy environment configuration (if needed)
cp config/envs/dev.env.example config/envs/dev.env
```

### 3. Database Setup
```bash
# SQLite (default) - database file created automatically
# Or for PostgreSQL, update connection string in config

# Run initial setup
./bin/build/app_dev.sh
```

### 4. Start Development Server
```bash
# Start with hot reloading
./bin/run/app_web_dev.sh

# Or build and run manually
./bin/build/app_dev.sh
./app
```

## Development Environment

### Hot Reloading Setup
The framework uses Air for automatic rebuilding during development:

```bash
# Start development server with hot reloading
./bin/run/app_web_dev.sh
```

This command:
- Watches Go files for changes
- Automatically rebuilds the application
- Restarts the server
- Compiles Tailwind CSS on file changes

### CSS Compilation
Tailwind CSS compilation runs in watch mode:

```bash
# Manual CSS compilation with watch
npx tailwindcss -i input.css -o ./assets/css/output.css --watch
```

## Docker Development

### Using Docker Compose
```bash
# Development environment
docker-compose up dev

# Production environment
docker-compose up prod
```

### Manual Docker Build
```bash
# Development build
docker build --target dev -t app-dev .

# Production build
docker build --target prod -t app-prod .
```

## Environment Variables

### Core Configuration
```bash
ENV=dev                    # Environment: dev, prod
PORT=8080                  # HTTP server port
DB_PATH=data/db.db        # SQLite database path
```

### PostgreSQL Configuration (Optional)
```bash
DB_HOST=localhost
DB_PORT=5432
DB_NAME=myapp
DB_USER=user
DB_PASS=password
```

## Verification

### Check Installation
```bash
# Verify Go build
go build -o app main.go

# Verify application starts
./app

# Check HTTP endpoint
curl http://localhost:8080
```

### Run Tests
```bash
# Run all tests
./bin/tests/check.sh

# Run specific test
go test ./tests/functional/path/to/specific_test.go
```

## Common Issues

### Go Module Issues
```bash
# Clean and reinstall modules
rm go.sum
go clean -modcache
go mod tidy
go mod vendor
```

### CSS Compilation Issues
```bash
# Reinstall Node dependencies
rm -rf node_modules package-lock.json
npm install

# Manual CSS build
npx tailwindcss -i input.css -o ./assets/css/output.css
```

### Database Issues
```bash
# Reset SQLite database
rm data/db.db
./app  # Will recreate database
```

## Next Steps

After successful installation:

1. **[Quick Start Guide](quickstart.md)** - Build your first application
2. **[First Application](first-application.md)** - Complete walkthrough
3. **[Configuration](configuration.md)** - Environment setup details

## LLM Integration Notes

When helping with installation issues:
- Check `ENV` environment variable first
- Verify all build scripts have execute permissions
- Database path is relative to project root
- Tailwind requires Node.js for compilation
- Air configuration is in `air.dev.toml`