# Handlers Service Kind

## Overview

Handlers are entity-specific services responsible for Create, Read, Update, Delete (CRUD) operations on domain models. They provide a standardized interface for data persistence operations and manage transaction handling, field selection, and automatic timestamp utilities.

## Architecture

### Core Components

**Base Handler Library** (`src/lib/handler/`)
- `Handler` struct: Core database operation executor with logging and timing
- `HandlerMod`: Operation configuration object containing transaction, table, field, and value specifications
- Database abstraction layer integration with `SqlDb`

**Entity-Specific Handlers** (`src/srv/handlers/`)
- Each domain model has its own dedicated handler (e.g., `GardenHandler`, `UserHandler`)
- All handlers follow the same interface pattern and method conventions
- Integrated with corresponding Table services for data mapping and field management

### Key Structures

#### HandlerMod
Operation configuration object that controls:
- **Transaction**: `Tx` field for database transaction management
- **Entity Identity**: `Id` field for targeting specific records
- **Table**: Target database table name
- **Field Selection**: `Fields` array for partial updates/operations
- **Data Mapping**: `Columns` and `Values` arrays for SQL operations

#### Handler Base Class
Provides core functionality:
- Database operation execution (Create/Update/Delete)
- Logging and monitoring through Spy integration
- Field indexing and subset operations
- Automatic timestamp management
- Transaction support

## Standard Methods

Every entity handler implements these standard CRUD methods:

### Create Operations
```go
Create(ctx, entity, mod) (*Entity, error)      // Create new record
MustCreate(ctx, entity, mod) *Entity           // Create with panic on error
```

### Update Operations
```go
Update(ctx, entity, mod) (*Entity, error)           // Update entire record
MustUpdate(ctx, entity, mod) *Entity                // Update with panic on error
UpdateFields(ctx, entity, fields, mod) (*Entity, error)  // Partial update
MustUpdateFields(ctx, entity, fields, mod) *Entity       // Partial update with panic
```

### Delete Operations
```go
Delete(ctx, entity, mod) error                 // Delete record
MustDelete(ctx, entity, mod)                   // Delete with panic on error
```

### Internal Methods
```go
modDefaulting(mod) *HandlerMod                 // Default mod configuration
setColumns(mod, entity)                        // Configure columns and values
setUtilities(mod, entity, action)             // Set timestamps and utilities
```

## Usage Examples

### Basic Entity Creation
```go
garden := &mdl.Garden{Name: "My Garden", Picture: "garden.jpg"}
createdGarden, err := gardenHandler.Create(ctx, garden, nil)
// garden.Id is automatically set after creation
```

### Entity Updates
```go
// Full update
garden.Name = "Updated Garden Name"
updatedGarden, err := gardenHandler.Update(ctx, garden, nil)

// Partial update (only specific fields)
garden.Picture = "new_picture.jpg"
updatedGarden, err := gardenHandler.UpdateFields(ctx, garden, []string{"Picture"}, nil)
```

### Entity Deletion
```go
err := gardenHandler.Delete(ctx, garden, nil)
```

### Transaction Support
```go
mod := &handler.HandlerMod{Tx: tx}  // Use existing transaction
garden, err := gardenHandler.Create(ctx, garden, mod)
```

## Automatic Utilities

Handlers automatically manage common database utilities:

### Timestamp Management
- **created_at**: Automatically set on record creation
- **updated_at**: Automatically set on both creation and updates
- Uses the framework's `Time` abstraction for consistent timing

### Field Selection
- Default behavior: operates on all entity fields
- Selective updates: specify `Fields` in `HandlerMod` for partial operations
- Automatic field-to-column mapping through Table integration

## Integration with Tables

Handlers work closely with Table services:
- Use `Table.TableName()` for SQL table identification
- Use `Table.Fields()` and `Table.Columns()` for field mapping
- Use `Table.EntityToValues()` for converting entities to SQL values
- Leverage field indexing for partial operations

## SQL Operation Building

The base Handler automatically:
- Constructs parameterized INSERT statements for creates
- Builds UPDATE statements with WHERE clauses for updates
- Generates DELETE statements with proper ID targeting
- Manages field subset operations for partial updates
- Handles transaction integration when provided

## Error Handling

- Standard methods return errors for caller handling
- Must* methods panic on errors (using `lib.Poe()`)
- Database constraint violations bubble up as errors
- Transaction rollback handling when operations fail

## Logging and Monitoring

Handlers integrate with the Spy logging system:
- All operations are logged with operation type and details
- `HandlerMod.Printable()` provides JSON serialization for logging
- Records include timing information for performance monitoring
- Operation tracking for debugging and auditing
