# Stringers Service Kind

## Overview

Stringers are entity-specific services responsible for converting domain model entities into human-readable string representations. They provide standardized string conversion for UI display, search indexing, logging, and other text-based operations requiring entity identification.

## Architecture

### Core Components

**Entity-Specific Stringers** (`src/srv/stringers/`)
- Each domain model has its own dedicated stringer (e.g., `GardenStringer`, `UserStringer`)
- All stringers follow the same interface pattern and method conventions
- No base library - stringers implement simple, focused string conversion logic

**Global Stringers Coordinator** (`src/srv/stringers/stringers.go`)
- `Stringers` struct provides polymorphic string conversion
- Type switching for entity-specific string conversion
- Centralized string conversion for any entity type

## Standard Methods

Every entity stringer implements these standard methods:

### String Conversion
```go
ToString(entity) (string, error)
```
- Convert entity to human-readable string representation
- Returns empty string for nil entities
- Handles conversion errors gracefully

### Must Conversion
```go
MustToString(entity) string
```
- Convert entity to string with panic on error
- Uses `lib.Poe()` for error handling
- Preferred for situations where conversion must succeed

## Usage Examples

### Basic String Conversion
```go
garden := &mdl.Garden{Name: "My Garden"}

// Safe conversion with error handling
str, err := gardenStringer.ToString(garden)
if err != nil {
    return err
}

// Must conversion (panics on error)
str := gardenStringer.MustToString(garden)
// Returns: "My Garden"
```

### Polymorphic Conversion
```go
// Convert any entity type to string
var entity interface{} = &mdl.Garden{Name: "Vegetable Garden"}

str, err := stringers.ToString(entity)
// Returns: "Vegetable Garden"
```

### UI Display Integration
```go
// In templates or UI components
garden := fetcher.MustFindOneById(ctx, id, nil)
displayName := gardenStringer.MustToString(garden)
// Use displayName in UI
```

## String Conversion Logic

Stringers implement entity-specific logic for meaningful string representation:

### Typical Field Selection
Most stringers use a primary identifying field:
```go
func (this *GardenStringer) ToString(garden *mdl.Garden) (string, error) {
    if garden == nil {
        return "", nil
    }
    return garden.Name, nil  // Use Name field as string representation
}
```

### Common Field Choices
- **Name fields**: Most common choice for entities with descriptive names
- **Title fields**: For entities with title-based identification
- **Code/Identifier fields**: For entities with business identifiers
- **Composite strings**: Combination of multiple fields when needed

## Nil Entity Handling

All stringers handle nil entities consistently:
```go
if entity == nil {
    return "", nil
}
```
- Returns empty string rather than panicking
- Allows safe usage in contexts where entities might be nil
- Consistent behavior across all entity stringers

## Global String Conversion

The `Stringers` struct provides type-agnostic string conversion:

### Type Switching
```go
func (this *Stringers) ToString(object interface{}) (string, error) {
    switch v := object.(type) {
    case *mdl.Garden:
        return this.GardenStringer.ToString(v)
    case *mdl.User:
        return this.UserStringer.ToString(v)
    // ... all entity types
    default:
        return "", errors.New("cannot convert the object into a string")
    }
}
```

### Usage Patterns
- **UI Components**: Display any entity type without knowing specific type
- **Logging**: Convert entities to strings for log messages
- **Search**: Generate searchable text from any entity
- **Generic Operations**: Handle mixed entity collections

## Integration with Framework

### With Searchers
```go
Text: strings.ToLower(this.GardenStringer.MustToString(entity))
```

### With Presenters
```go
displayName := this.GardenStringer.MustToString(garden)
// Use in HTML templates or UI components
```

### With Logging
```go
logger.Info("Processing garden", "garden", gardenStringer.MustToString(garden))
```

## Error Handling

### ToString Method
- Returns error for conversion failures (rare)
- Returns empty string for nil entities (not an error)
- Graceful handling of edge cases

### MustToString Method
- Panics on conversion errors using `lib.Poe()`
- Suitable for situations where string conversion must succeed
- Used when nil entities have already been filtered out

## Use Cases

### UI Display
- Form labels and dropdowns
- List item display text
- Breadcrumb navigation
- Entity references in views

### Search and Indexing
- Generate searchable text content
- Create search result summaries
- Index entity names for autocomplete

### Logging and Debugging
- Entity identification in log messages
- Debug output for entity processing
- Error reporting with entity context

### API Responses
- Human-readable entity identification
- Reference display in JSON responses
- Entity summaries for client applications