package fetchers

// |@@| C

import (
	"context"
	"database/sql"
	"gardening/src/lib"
	"gardening/src/lib/fetcher"
	"gardening/src/mdl"
	"gardening/src/srv/tables"
)

type GardenFetcher struct {
	Fetcher     *fetcher.Fetcher
	GardenTable *tables.GardenTable
}

func (this *GardenFetcher) FindAll(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Garden, error) {
	mod = this.modDefaulting(mod)
	return this.querySet(ctx, mod)
}
func (this *GardenFetcher) FindSet(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Garden, error) {
	mod = this.modDefaulting(mod)
	return this.querySet(ctx, mod)
}
func (this *GardenFetcher) FindSetIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.Garden, error) {
	mod = this.modDefaulting(mod)
	mod.InStringValuesFilter("id", ids)
	return this.FindSet(ctx, mod)
}
func (this *GardenFetcher) FindPageIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.Garden, *fetcher.FetcherPagination, error) {
	mod = this.modDefaulting(mod)
	mod.InStringValuesFilter("id", ids)
	return this.FindPage(ctx, mod)
}
func (this *GardenFetcher) FindPage(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Garden, *fetcher.FetcherPagination, error) {
	mod = this.modDefaulting(mod)
	set, err := this.queryPage(ctx, mod)
	if err != nil {
		return nil, nil, err
	}
	pagination := this.CountSet(ctx, mod)
	pagination.Modulation = mod
	return set, pagination, nil
}
func (this *GardenFetcher) FindOne(ctx context.Context, mod *fetcher.FetcherMod) (*mdl.Garden, bool, error) {
	mod = this.modDefaulting(mod)
	return this.queryOne(ctx, mod)
}
func (this *GardenFetcher) ShallFindOne(ctx context.Context, mod *fetcher.FetcherMod) *mdl.Garden {
	mod = this.modDefaulting(mod)
	p1, has, err := this.FindOne(ctx, mod)
	lib.Poe(err)
	this.Fetcher.MustHave(has, mod)
	return p1
}
func (this *GardenFetcher) FindOneById(ctx context.Context, id string, mod *fetcher.FetcherMod) (*mdl.Garden, bool, error) {
	mod = this.modDefaulting(mod)
	mod.ExactStringValueFilter("id", id)
	return this.FindOne(ctx, mod)
}
func (this *GardenFetcher) ShallFindOneById(ctx context.Context, id string, mod *fetcher.FetcherMod) *mdl.Garden {
	mod = this.modDefaulting(mod)
	p1, has, err := this.FindOneById(ctx, id, mod)
	lib.Poe(err)
	this.Fetcher.MustHave(has, mod)
	return p1
}
func (this *GardenFetcher) Mod() *fetcher.FetcherMod {
	return &fetcher.FetcherMod{
		Page:    1,
		PerPage: 25,
		DefaultOrder: &fetcher.FetcherOrderBy{
			Order:  "ASC",
			Column: "name",
		},
	}
}
func (this *GardenFetcher) buildSelect(ctx context.Context, mod *fetcher.FetcherMod) (string, []interface{}) {
	return this.Fetcher.BuildSelect(mod, this.GardenTable.TableName(), this.GardenTable.Columns())
}
func (this *GardenFetcher) buildCount(ctx context.Context, mod *fetcher.FetcherMod) (string, []interface{}) {
	mod.Count = true
	return this.Fetcher.BuildSelect(mod, this.GardenTable.TableName(), this.GardenTable.Columns())
}
func (this *GardenFetcher) queryOne(ctx context.Context, mod *fetcher.FetcherMod) (*mdl.Garden, bool, error) {
	mod.One = true
	query, args := this.buildSelect(ctx, mod)
	eso := &tables.GardenEso{}
	row := this.Fetcher.QueryRow(ctx, query, args...)
	err := row.Scan(this.GardenTable.EsoToPointers(eso)...)
	if this.Fetcher.IsNoRow(err) {
		return nil, false, nil
	}
	lib.Poe(err)
	entity := this.GardenTable.EsoToEntity(eso)
	return entity, true, nil
}
func (this *GardenFetcher) querySet(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Garden, error) {
	query, args := this.buildSelect(ctx, mod)
	rows, err := this.Fetcher.Query(ctx, query, args...)
	lib.Poe(err)
	set := this.rowsScan(ctx, rows, mod)
	err = rows.Close()
	lib.Poe(err)
	return set, nil
}
func (this *GardenFetcher) queryPage(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Garden, error) {
	mod.Paginated = true
	return this.querySet(ctx, mod)
}
func (this *GardenFetcher) CountSet(ctx context.Context, mod *fetcher.FetcherMod) *fetcher.FetcherPagination {
	pagination := &fetcher.FetcherPagination{}
	query, args := this.buildCount(ctx, mod)
	row := this.Fetcher.QueryRow(ctx, query, args...)
	err := row.Scan(&pagination.Total)
	lib.Poe(err)
	this.Fetcher.CalculatePagination(pagination, mod)
	return pagination
}
func (this *GardenFetcher) rowsScan(ctx context.Context, rows *sql.Rows, mod *fetcher.FetcherMod) []*mdl.Garden {
	var set []*mdl.Garden
	for rows.Next() {
		eso := &tables.GardenEso{}
		fields := this.GardenTable.EsoToPointers(eso)
		err := rows.Scan(fields...)
		lib.Poe(err)
		entity := this.GardenTable.EsoToEntity(eso)
		set = append(set, entity)
	}
	err := rows.Close()
	lib.Poe(err)
	return set
}
func (this *GardenFetcher) modDefaulting(mod *fetcher.FetcherMod) *fetcher.FetcherMod {
	if mod == nil {
		return this.Mod()
	}
	return mod
}
func (this *GardenFetcher) MustFindAll(ctx context.Context, mod *fetcher.FetcherMod) []*mdl.Garden {
	p1, err := this.FindAll(ctx, mod)
	lib.Poe(err)
	return p1
}
func (this *GardenFetcher) MustFindSet(ctx context.Context, mod *fetcher.FetcherMod) []*mdl.Garden {
	p1, err := this.FindSet(ctx, mod)
	lib.Poe(err)
	return p1
}
func (this *GardenFetcher) MustFindSetIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) []*mdl.Garden {
	p1, err := this.FindSetIn(ctx, ids, mod)
	lib.Poe(err)
	return p1
}
func (this *GardenFetcher) MustFindPageIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.Garden, *fetcher.FetcherPagination) {
	p1, p2, err := this.FindPageIn(ctx, ids, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *GardenFetcher) MustFindPage(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Garden, *fetcher.FetcherPagination) {
	p1, p2, err := this.FindPage(ctx, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *GardenFetcher) MustFindOne(ctx context.Context, mod *fetcher.FetcherMod) (*mdl.Garden, bool) {
	p1, p2, err := this.FindOne(ctx, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *GardenFetcher) MustFindOneById(ctx context.Context, id string, mod *fetcher.FetcherMod) (*mdl.Garden, bool) {
	p1, p2, err := this.FindOneById(ctx, id, mod)
	lib.Poe(err)
	return p1, p2
}
