package fetchers

// |@@| C

import (
	"context"
	"database/sql"
	"gardening/src/lib"
	"gardening/src/lib/fetcher"
	"gardening/src/mdl"
	"gardening/src/srv/tables"
)

type GardenTaskFetcher struct {
	Fetcher         *fetcher.Fetcher
	GardenTaskTable *tables.GardenTaskTable
}

func (this *GardenTaskFetcher) FindAll(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, error) {
	mod = this.modDefaulting(mod)
	return this.querySet(ctx, mod)
}
func (this *GardenTaskFetcher) FindSet(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, error) {
	mod = this.modDefaulting(mod)
	return this.querySet(ctx, mod)
}
func (this *GardenTaskFetcher) FindSetIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, error) {
	mod = this.modDefaulting(mod)
	mod.InStringValuesFilter("id", ids)
	return this.FindSet(ctx, mod)
}
func (this *GardenTaskFetcher) FindPageIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, *fetcher.FetcherPagination, error) {
	mod = this.modDefaulting(mod)
	mod.InStringValuesFilter("id", ids)
	return this.FindPage(ctx, mod)
}
func (this *GardenTaskFetcher) FindSetInGardenIds(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, error) {
	mod = this.modDefaulting(mod)
	mod.InStringValuesFilter("garden_id", ids)
	return this.FindSet(ctx, mod)
}
func (this *GardenTaskFetcher) FindSetInPlantationIds(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, error) {
	mod = this.modDefaulting(mod)
	mod.InStringValuesFilter("plantation_id", ids)
	return this.FindSet(ctx, mod)
}
func (this *GardenTaskFetcher) FindPage(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, *fetcher.FetcherPagination, error) {
	mod = this.modDefaulting(mod)
	set, err := this.queryPage(ctx, mod)
	if err != nil {
		return nil, nil, err
	}
	pagination := this.CountSet(ctx, mod)
	pagination.Modulation = mod
	return set, pagination, nil
}
func (this *GardenTaskFetcher) FindOne(ctx context.Context, mod *fetcher.FetcherMod) (*mdl.GardenTask, bool, error) {
	mod = this.modDefaulting(mod)
	return this.queryOne(ctx, mod)
}
func (this *GardenTaskFetcher) ShallFindOne(ctx context.Context, mod *fetcher.FetcherMod) *mdl.GardenTask {
	mod = this.modDefaulting(mod)
	p1, has, err := this.FindOne(ctx, mod)
	lib.Poe(err)
	this.Fetcher.MustHave(has, mod)
	return p1
}
func (this *GardenTaskFetcher) FindOneById(ctx context.Context, id string, mod *fetcher.FetcherMod) (*mdl.GardenTask, bool, error) {
	mod = this.modDefaulting(mod)
	mod.ExactStringValueFilter("id", id)
	return this.FindOne(ctx, mod)
}
func (this *GardenTaskFetcher) ShallFindOneById(ctx context.Context, id string, mod *fetcher.FetcherMod) *mdl.GardenTask {
	mod = this.modDefaulting(mod)
	p1, has, err := this.FindOneById(ctx, id, mod)
	lib.Poe(err)
	this.Fetcher.MustHave(has, mod)
	return p1
}
func (this *GardenTaskFetcher) FindSetByGardenId(ctx context.Context, gardenId string, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, error) {
	mod = this.modDefaulting(mod)
	mod.ExactStringValueFilter("garden_id", gardenId)
	return this.FindSet(ctx, mod)
}
func (this *GardenTaskFetcher) FindSetByPlantationId(ctx context.Context, plantationId string, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, error) {
	mod = this.modDefaulting(mod)
	mod.ExactStringValueFilter("plantation_id", plantationId)
	return this.FindSet(ctx, mod)
}
func (this *GardenTaskFetcher) Mod() *fetcher.FetcherMod {
	return &fetcher.FetcherMod{
		Page:    1,
		PerPage: 25,
	}
}
func (this *GardenTaskFetcher) buildSelect(ctx context.Context, mod *fetcher.FetcherMod) (string, []interface{}) {
	return this.Fetcher.BuildSelect(mod, this.GardenTaskTable.TableName(), this.GardenTaskTable.Columns())
}
func (this *GardenTaskFetcher) buildCount(ctx context.Context, mod *fetcher.FetcherMod) (string, []interface{}) {
	mod.Count = true
	return this.Fetcher.BuildSelect(mod, this.GardenTaskTable.TableName(), this.GardenTaskTable.Columns())
}
func (this *GardenTaskFetcher) queryOne(ctx context.Context, mod *fetcher.FetcherMod) (*mdl.GardenTask, bool, error) {
	mod.One = true
	query, args := this.buildSelect(ctx, mod)
	eso := &tables.GardenTaskEso{}
	row := this.Fetcher.QueryRow(ctx, query, args...)
	err := row.Scan(this.GardenTaskTable.EsoToPointers(eso)...)
	if this.Fetcher.IsNoRow(err) {
		return nil, false, nil
	}
	lib.Poe(err)
	entity := this.GardenTaskTable.EsoToEntity(eso)
	return entity, true, nil
}
func (this *GardenTaskFetcher) querySet(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, error) {
	query, args := this.buildSelect(ctx, mod)
	rows, err := this.Fetcher.Query(ctx, query, args...)
	lib.Poe(err)
	set := this.rowsScan(ctx, rows, mod)
	err = rows.Close()
	lib.Poe(err)
	return set, nil
}
func (this *GardenTaskFetcher) queryPage(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, error) {
	mod.Paginated = true
	return this.querySet(ctx, mod)
}
func (this *GardenTaskFetcher) CountSet(ctx context.Context, mod *fetcher.FetcherMod) *fetcher.FetcherPagination {
	pagination := &fetcher.FetcherPagination{}
	query, args := this.buildCount(ctx, mod)
	row := this.Fetcher.QueryRow(ctx, query, args...)
	err := row.Scan(&pagination.Total)
	lib.Poe(err)
	this.Fetcher.CalculatePagination(pagination, mod)
	return pagination
}
func (this *GardenTaskFetcher) rowsScan(ctx context.Context, rows *sql.Rows, mod *fetcher.FetcherMod) []*mdl.GardenTask {
	var set []*mdl.GardenTask
	for rows.Next() {
		eso := &tables.GardenTaskEso{}
		fields := this.GardenTaskTable.EsoToPointers(eso)
		err := rows.Scan(fields...)
		lib.Poe(err)
		entity := this.GardenTaskTable.EsoToEntity(eso)
		set = append(set, entity)
	}
	err := rows.Close()
	lib.Poe(err)
	return set
}
func (this *GardenTaskFetcher) modDefaulting(mod *fetcher.FetcherMod) *fetcher.FetcherMod {
	if mod == nil {
		return this.Mod()
	}
	return mod
}
func (this *GardenTaskFetcher) MustFindAll(ctx context.Context, mod *fetcher.FetcherMod) []*mdl.GardenTask {
	p1, err := this.FindAll(ctx, mod)
	lib.Poe(err)
	return p1
}
func (this *GardenTaskFetcher) MustFindSet(ctx context.Context, mod *fetcher.FetcherMod) []*mdl.GardenTask {
	p1, err := this.FindSet(ctx, mod)
	lib.Poe(err)
	return p1
}
func (this *GardenTaskFetcher) MustFindSetIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) []*mdl.GardenTask {
	p1, err := this.FindSetIn(ctx, ids, mod)
	lib.Poe(err)
	return p1
}
func (this *GardenTaskFetcher) MustFindPageIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, *fetcher.FetcherPagination) {
	p1, p2, err := this.FindPageIn(ctx, ids, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *GardenTaskFetcher) MustFindSetInGardenIds(ctx context.Context, ids []string, mod *fetcher.FetcherMod) []*mdl.GardenTask {
	p1, err := this.FindSetInGardenIds(ctx, ids, mod)
	lib.Poe(err)
	return p1
}
func (this *GardenTaskFetcher) MustFindSetInPlantationIds(ctx context.Context, ids []string, mod *fetcher.FetcherMod) []*mdl.GardenTask {
	p1, err := this.FindSetInPlantationIds(ctx, ids, mod)
	lib.Poe(err)
	return p1
}
func (this *GardenTaskFetcher) MustFindPage(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.GardenTask, *fetcher.FetcherPagination) {
	p1, p2, err := this.FindPage(ctx, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *GardenTaskFetcher) MustFindOne(ctx context.Context, mod *fetcher.FetcherMod) (*mdl.GardenTask, bool) {
	p1, p2, err := this.FindOne(ctx, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *GardenTaskFetcher) MustFindOneById(ctx context.Context, id string, mod *fetcher.FetcherMod) (*mdl.GardenTask, bool) {
	p1, p2, err := this.FindOneById(ctx, id, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *GardenTaskFetcher) MustFindSetByGardenId(ctx context.Context, gardenId string, mod *fetcher.FetcherMod) []*mdl.GardenTask {
	p1, err := this.FindSetByGardenId(ctx, gardenId, mod)
	lib.Poe(err)
	return p1
}
func (this *GardenTaskFetcher) MustFindSetByPlantationId(ctx context.Context, plantationId string, mod *fetcher.FetcherMod) []*mdl.GardenTask {
	p1, err := this.FindSetByPlantationId(ctx, plantationId, mod)
	lib.Poe(err)
	return p1
}
