package fetchers

// |@@| C

import (
	"context"
	"database/sql"
	"gardening/src/lib"
	"gardening/src/lib/fetcher"
	"gardening/src/mdl"
	"gardening/src/srv/tables"
)

type PlantationFetcher struct {
	Fetcher         *fetcher.Fetcher
	PlantationTable *tables.PlantationTable
}

func (this *PlantationFetcher) FindAll(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Plantation, error) {
	mod = this.modDefaulting(mod)
	return this.querySet(ctx, mod)
}
func (this *PlantationFetcher) FindSet(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Plantation, error) {
	mod = this.modDefaulting(mod)
	return this.querySet(ctx, mod)
}
func (this *PlantationFetcher) FindSetIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.Plantation, error) {
	mod = this.modDefaulting(mod)
	mod.InStringValuesFilter("id", ids)
	return this.FindSet(ctx, mod)
}
func (this *PlantationFetcher) FindPageIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.Plantation, *fetcher.FetcherPagination, error) {
	mod = this.modDefaulting(mod)
	mod.InStringValuesFilter("id", ids)
	return this.FindPage(ctx, mod)
}
func (this *PlantationFetcher) FindSetInPlantIds(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.Plantation, error) {
	mod = this.modDefaulting(mod)
	mod.InStringValuesFilter("plant_id", ids)
	return this.FindSet(ctx, mod)
}
func (this *PlantationFetcher) FindSetInGardenIds(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.Plantation, error) {
	mod = this.modDefaulting(mod)
	mod.InStringValuesFilter("garden_id", ids)
	return this.FindSet(ctx, mod)
}
func (this *PlantationFetcher) FindPage(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Plantation, *fetcher.FetcherPagination, error) {
	mod = this.modDefaulting(mod)
	set, err := this.queryPage(ctx, mod)
	if err != nil {
		return nil, nil, err
	}
	pagination := this.CountSet(ctx, mod)
	pagination.Modulation = mod
	return set, pagination, nil
}
func (this *PlantationFetcher) FindOne(ctx context.Context, mod *fetcher.FetcherMod) (*mdl.Plantation, bool, error) {
	mod = this.modDefaulting(mod)
	return this.queryOne(ctx, mod)
}
func (this *PlantationFetcher) ShallFindOne(ctx context.Context, mod *fetcher.FetcherMod) *mdl.Plantation {
	mod = this.modDefaulting(mod)
	p1, has, err := this.FindOne(ctx, mod)
	lib.Poe(err)
	this.Fetcher.MustHave(has, mod)
	return p1
}
func (this *PlantationFetcher) FindOneById(ctx context.Context, id string, mod *fetcher.FetcherMod) (*mdl.Plantation, bool, error) {
	mod = this.modDefaulting(mod)
	mod.ExactStringValueFilter("id", id)
	return this.FindOne(ctx, mod)
}
func (this *PlantationFetcher) ShallFindOneById(ctx context.Context, id string, mod *fetcher.FetcherMod) *mdl.Plantation {
	mod = this.modDefaulting(mod)
	p1, has, err := this.FindOneById(ctx, id, mod)
	lib.Poe(err)
	this.Fetcher.MustHave(has, mod)
	return p1
}
func (this *PlantationFetcher) FindSetByPlantId(ctx context.Context, plantId string, mod *fetcher.FetcherMod) ([]*mdl.Plantation, error) {
	mod = this.modDefaulting(mod)
	mod.ExactStringValueFilter("plant_id", plantId)
	return this.FindSet(ctx, mod)
}
func (this *PlantationFetcher) FindSetByGardenId(ctx context.Context, gardenId string, mod *fetcher.FetcherMod) ([]*mdl.Plantation, error) {
	mod = this.modDefaulting(mod)
	mod.ExactStringValueFilter("garden_id", gardenId)
	return this.FindSet(ctx, mod)
}
func (this *PlantationFetcher) Mod() *fetcher.FetcherMod {
	return &fetcher.FetcherMod{
		Page:    1,
		PerPage: 25,
	}
}
func (this *PlantationFetcher) buildSelect(ctx context.Context, mod *fetcher.FetcherMod) (string, []interface{}) {
	return this.Fetcher.BuildSelect(mod, this.PlantationTable.TableName(), this.PlantationTable.Columns())
}
func (this *PlantationFetcher) buildCount(ctx context.Context, mod *fetcher.FetcherMod) (string, []interface{}) {
	mod.Count = true
	return this.Fetcher.BuildSelect(mod, this.PlantationTable.TableName(), this.PlantationTable.Columns())
}
func (this *PlantationFetcher) queryOne(ctx context.Context, mod *fetcher.FetcherMod) (*mdl.Plantation, bool, error) {
	mod.One = true
	query, args := this.buildSelect(ctx, mod)
	eso := &tables.PlantationEso{}
	row := this.Fetcher.QueryRow(ctx, query, args...)
	err := row.Scan(this.PlantationTable.EsoToPointers(eso)...)
	if this.Fetcher.IsNoRow(err) {
		return nil, false, nil
	}
	lib.Poe(err)
	entity := this.PlantationTable.EsoToEntity(eso)
	return entity, true, nil
}
func (this *PlantationFetcher) querySet(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Plantation, error) {
	query, args := this.buildSelect(ctx, mod)
	rows, err := this.Fetcher.Query(ctx, query, args...)
	lib.Poe(err)
	set := this.rowsScan(ctx, rows, mod)
	err = rows.Close()
	lib.Poe(err)
	return set, nil
}
func (this *PlantationFetcher) queryPage(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Plantation, error) {
	mod.Paginated = true
	return this.querySet(ctx, mod)
}
func (this *PlantationFetcher) CountSet(ctx context.Context, mod *fetcher.FetcherMod) *fetcher.FetcherPagination {
	pagination := &fetcher.FetcherPagination{}
	query, args := this.buildCount(ctx, mod)
	row := this.Fetcher.QueryRow(ctx, query, args...)
	err := row.Scan(&pagination.Total)
	lib.Poe(err)
	this.Fetcher.CalculatePagination(pagination, mod)
	return pagination
}
func (this *PlantationFetcher) rowsScan(ctx context.Context, rows *sql.Rows, mod *fetcher.FetcherMod) []*mdl.Plantation {
	var set []*mdl.Plantation
	for rows.Next() {
		eso := &tables.PlantationEso{}
		fields := this.PlantationTable.EsoToPointers(eso)
		err := rows.Scan(fields...)
		lib.Poe(err)
		entity := this.PlantationTable.EsoToEntity(eso)
		set = append(set, entity)
	}
	err := rows.Close()
	lib.Poe(err)
	return set
}
func (this *PlantationFetcher) modDefaulting(mod *fetcher.FetcherMod) *fetcher.FetcherMod {
	if mod == nil {
		return this.Mod()
	}
	return mod
}
func (this *PlantationFetcher) MustFindAll(ctx context.Context, mod *fetcher.FetcherMod) []*mdl.Plantation {
	p1, err := this.FindAll(ctx, mod)
	lib.Poe(err)
	return p1
}
func (this *PlantationFetcher) MustFindSet(ctx context.Context, mod *fetcher.FetcherMod) []*mdl.Plantation {
	p1, err := this.FindSet(ctx, mod)
	lib.Poe(err)
	return p1
}
func (this *PlantationFetcher) MustFindSetIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) []*mdl.Plantation {
	p1, err := this.FindSetIn(ctx, ids, mod)
	lib.Poe(err)
	return p1
}
func (this *PlantationFetcher) MustFindPageIn(ctx context.Context, ids []string, mod *fetcher.FetcherMod) ([]*mdl.Plantation, *fetcher.FetcherPagination) {
	p1, p2, err := this.FindPageIn(ctx, ids, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *PlantationFetcher) MustFindSetInPlantIds(ctx context.Context, ids []string, mod *fetcher.FetcherMod) []*mdl.Plantation {
	p1, err := this.FindSetInPlantIds(ctx, ids, mod)
	lib.Poe(err)
	return p1
}
func (this *PlantationFetcher) MustFindSetInGardenIds(ctx context.Context, ids []string, mod *fetcher.FetcherMod) []*mdl.Plantation {
	p1, err := this.FindSetInGardenIds(ctx, ids, mod)
	lib.Poe(err)
	return p1
}
func (this *PlantationFetcher) MustFindPage(ctx context.Context, mod *fetcher.FetcherMod) ([]*mdl.Plantation, *fetcher.FetcherPagination) {
	p1, p2, err := this.FindPage(ctx, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *PlantationFetcher) MustFindOne(ctx context.Context, mod *fetcher.FetcherMod) (*mdl.Plantation, bool) {
	p1, p2, err := this.FindOne(ctx, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *PlantationFetcher) MustFindOneById(ctx context.Context, id string, mod *fetcher.FetcherMod) (*mdl.Plantation, bool) {
	p1, p2, err := this.FindOneById(ctx, id, mod)
	lib.Poe(err)
	return p1, p2
}
func (this *PlantationFetcher) MustFindSetByPlantId(ctx context.Context, plantId string, mod *fetcher.FetcherMod) []*mdl.Plantation {
	p1, err := this.FindSetByPlantId(ctx, plantId, mod)
	lib.Poe(err)
	return p1
}
func (this *PlantationFetcher) MustFindSetByGardenId(ctx context.Context, gardenId string, mod *fetcher.FetcherMod) []*mdl.Plantation {
	p1, err := this.FindSetByGardenId(ctx, gardenId, mod)
	lib.Poe(err)
	return p1
}
